package mcfall.math;

/**
 * A ColumnVector is a Vector whose values are stored in a single column.  It is therefore an mx1 Matrix
 * @author mcfall
 *
 */
public class ColumnVector extends Vector {
	
	/*  
	 * Constructs a column vector having the specified number of rows
	 * @param numRows a positive integer indicating the number of columns for this ColumnVector
	 */
	public ColumnVector (int numRows) {
		super (numRows, 1);
	}
	
	/*
	 * Constructs a column vector with the specified number of rows and the specified initial values.  If
	 * <i>initialValues</i> has fewer than numRows values, the first n components of the vector will be
	 * initialized, and the remaining components will be 0.  If initialValues contains more than numRows
	 * values, the first numRows values will be used to initialize the vector, and the remaining values
	 * will be ignored
	 * @param numRows a positive integer specifying the number of rows in this ColumnVector
	 * @param initialValues an array of double values used to initialize the vector components
	 * 
	 */
	public ColumnVector (int numRows, double[] initialValues) {
		super (numRows, 1, initialValues);
	}

	/**
	 * Constructs a 4-value Column vector that travels in the direction needed to go from the point
	 * <i>from</i> to the point <i>to</i>
	 * @param from the starting location of the vector
	 * @param to the ending location of the vector
	 */
	public ColumnVector (Point from, Point to) {
		this(4);
		setValueAt(1, to.getX()-from.getX());
		setValueAt(2, to.getY()-from.getY());
		setValueAt(3, to.getZ()-from.getZ());
	}
	
	@Override
	public double setValueAt(int i, double value) throws IndexOutOfBoundsException {
		return super.setValueAt (i, getFirstColumnIndex(), value);		
	}

	@Override
	public double getValueAt(int i) throws IndexOutOfBoundsException {
		return super.getValueAt(i, getFirstColumnIndex());
	}

	@Override
	public Vector duplicate() {
		ColumnVector copy = new ColumnVector (getNumberOfRows());
		copy.setFirstColumnIndex(getFirstColumnIndex());
		copy.setFirstRowIndex(getFirstRowIndex());
		for (int row = getFirstRowIndex(); row <= getLastRowIndex(); row++) {
			copy.setValueAt(row, getValueAt(row));
		}
		return copy;
	}

	@Override
	public int getFirstIndex() {
		return getFirstRowIndex();
	}

	@Override
	public int getSize() {
		return getNumberOfRows();
	}
}
